<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Config;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class CategoryTemplate extends Model
{
    use HasFactory;

    protected $table = 'category_templates';

    /**
     * فیلدهای قابل پر شدن
     */
    protected $fillable = [
        'category_id',
        'template_name',
        'template_key',
        'template_type',
        'parent_template_id',
        'header_style',
        'content_layout',
        'primary_color',
        'secondary_color',
        'icon_set',
        'background_image',
        'features',
        'feature_settings',
        'custom_css',
        'custom_js',
        'is_active'
    ];

    /**
     * فیلدهای آرایه‌ای
     */
    protected $casts = [
        'features' => 'array',
        'feature_settings' => 'array',
        'is_active' => 'boolean'
    ];

    /**
     * رابطه با دسته‌بندی
     */
    public function category(): BelongsTo
    {
        return $this->belongsTo(Category::class);
    }

    /**
     * رابطه با قالب والد
     */
    public function parentTemplate(): BelongsTo
    {
        return $this->belongsTo(CategoryTemplate::class, 'parent_template_id');
    }

    /**
     * رابطه با قالب‌های فرزند
     */
    public function childTemplates(): HasMany
    {
        return $this->hasMany(CategoryTemplate::class, 'parent_template_id');
    }

    /**
     * دریافت تنظیمات ویژگی
     */
    public function getFeatureSettings(string $feature): array
    {
        return $this->feature_settings[$feature] ?? config('templates.features.' . $feature . '.default_settings', []);
    }

    /**
     * دریافت ویژگی‌های فعال
     */
    public function getEnabledFeatures(): array
    {
        return $this->features ?? [];
    }

    /**
     * بررسی فعال بودن یک ویژگی خاص
     */
    public function isFeatureEnabled(string $feature, string $setting = 'is_enabled'): bool
    {
        if (!is_array($this->features)) {
            return false;
        }

        return isset($this->features[$feature][$setting]) &&
               $this->features[$feature][$setting] === true;
    }

    /**
     * دریافت سلسله مراتب قالب
     */
    public function getTemplateHierarchy(): array
    {
        $hierarchy = [$this];
        $current = $this;

        while ($current->parentTemplate) {
            $current = $current->parentTemplate;
            array_unshift($hierarchy, $current);
        }

        return $hierarchy;
    }

    /**
     * دریافت ویژگی‌های ارث‌بری شده
     */
    public function getInheritedFeatures(): array
    {
        $features = [];
        $hierarchy = $this->getTemplateHierarchy();

        foreach ($hierarchy as $template) {
            $features = array_merge($features, $template->getEnabledFeatures());
        }

        return array_unique($features);
    }

    /**
     * بررسی وجود قالب
     */
    public static function exists(int $id): bool
    {
        return static::where('id', $id)->exists();
    }

    /**
     * دریافت تنظیمات پیش‌فرض
     */
    public static function getDefaultSettings(): array
    {
        return [
            'template_type' => array_key_first(config('templates.types')),
            'header_style' => array_key_first(config('templates.header_styles')),
            'content_layout' => array_key_first(config('templates.content_layouts')),
            'primary_color' => '#000000',
            'secondary_color' => '#ffffff',
            'icon_set' => array_key_first(config('templates.icon_sets')),
            'features' => [],
            'feature_settings' => [],
            'is_active' => true
        ];
    }

    /**
     * Get the feature settings for a specific feature
     */
    public function getFeatureSettingsForTemplate($feature)
    {
        if (!isset($this->feature_settings[$feature])) {
            return Config::get("templates.features.{$feature}.settings", []);
        }

        return array_merge(
            Config::get("templates.features.{$feature}.settings", []),
            $this->feature_settings[$feature] ?? []
        );
    }

    /**
     * Get all enabled features in the template hierarchy
     */
    public function getInheritedFeatureSettings()
    {
        if (!$this->parent_template_id) {
            return $this->feature_settings ?? [];
        }

        $parentSettings = $this->parentTemplate->feature_settings ?? [];
        $currentSettings = $this->feature_settings ?? [];

        return array_merge($parentSettings, $currentSettings);
    }

    /**
     * Get the template view name
     */
    public function getTemplateViewName(): string
    {
        return 'templates.locations.' . $this->template_type;
    }

    /**
     * Get the template's CSS classes
     */
    public function getCssClasses(): string
    {
        $classes = [
            'template-' . $this->template_type,
            'header-' . $this->header_style,
            'layout-' . $this->content_layout
        ];

        return implode(' ', $classes);
    }

    /**
     * Get the template path
     */
    public function getTemplatePath()
    {
        return "templates.categories.{$this->template_type}";
    }

    /**
     * Get the header path
     */
    public function getHeaderPath()
    {
        return "templates.shared.components.header.{$this->header_style}";
    }

    /**
     * Get the footer path
     */
    public function getFooterPath()
    {
        return "templates.shared.components.footer.{$this->header_style}";
    }

    /**
     * Check if the template exists
     */
    public function templateExists()
    {
        return view()->exists($this->getTemplatePath());
    }

    /**
     * Create a new template with default settings
     */
    public static function createWithDefaults(array $attributes)
    {
        $defaults = static::getDefaultSettings();
        return static::create(array_merge($defaults, $attributes));
    }
}
