<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Services\SiteSettingService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class SiteSettingController extends Controller
{
    protected $settingService;

    public function __construct(SiteSettingService $settingService)
    {
        $this->settingService = $settingService;
    }

    /**
     * نمایش صفحه تنظیمات
     *
     * @return \Illuminate\View\View
     */
    public function index()
    {
        $settings = $this->settingService->getSettings();
        return view('admin.settings.index', compact('settings'));
    }

    /**
     * نمایش صفحه تنظیمات عمومی
     *
     * @return \Illuminate\View\View
     */
    public function general()
    {
        $settings = $this->settingService->getSettings();
        return view('admin.settings.general', compact('settings'));
    }

    /**
     * نمایش صفحه تنظیمات ایمیل
     *
     * @return \Illuminate\View\View
     */
    public function email()
    {
        $settings = $this->settingService->getSettings();
        return view('admin.settings.email', compact('settings'));
    }

    /**
     * نمایش صفحه تنظیمات پرداخت
     *
     * @return \Illuminate\View\View
     */
    public function payment()
    {
        $settings = $this->settingService->getSettings();
        return view('admin.settings.payment', compact('settings'));
    }

    /**
     * ذخیره تنظیمات جدید
     *
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(Request $request)
    {
        $request->validate([
            'site_title' => 'required|string|max:255',
            'site_icon' => 'nullable|image|mimes:jpeg,png,jpg,gif,ico|max:2048',
            'favicon' => 'nullable|image|mimes:jpeg,png,jpg,gif,ico|max:2048',
            'logo' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'logo_text' => 'nullable|string|max:255',
            'meta_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string|max:500',
            'meta_keywords' => 'nullable|string|max:500',
            'footer_text' => 'nullable|string',
            'contact_email' => 'nullable|email|max:255',
            'contact_phone' => 'nullable|string|max:20',
            'contact_address' => 'nullable|string|max:500',
            'theme_color' => 'nullable|string|max:7',
            'custom_css' => 'nullable|string',
            'custom_js' => 'nullable|string',
            'google_analytics_id' => 'nullable|string|max:50',
            'google_maps_api_key' => 'nullable|string|max:100',
            'maintenance_message' => 'nullable|string|max:500',
        ]);

        $data = $request->except(['_token', 'remove_site_icon', 'remove_favicon', 'remove_logo']);

        // پردازش فایل‌های آپلود شده
        $fileFields = ['site_icon', 'favicon', 'logo'];
        foreach ($fileFields as $field) {
            if ($request->hasFile($field)) {
                $data[$field] = $this->uploadFile($request->file($field), $field);
            } elseif ($request->has("remove_{$field}")) {
                // حذف فایل قبلی
                $oldFile = $this->settingService->getSetting($field);
                if ($oldFile) {
                    $this->deleteFile($oldFile);
                }
                $data[$field] = null;
            }
        }

        // ذخیره تنظیمات
        $this->settingService->saveSettings($data, app()->getLocale());

        // پاک کردن کش
        $this->settingService->clearCache();

        return redirect()->route('admin.settings.index')
            ->with('success', 'تنظیمات با موفقیت ذخیره شد.');
    }

    /**
     * آپلود فایل
     *
     * @param \Illuminate\Http\UploadedFile $file
     * @param string $field
     * @return string
     */
    protected function uploadFile($file, $field)
    {
        // حذف فایل قبلی اگر وجود داشته باشد
        $oldFile = $this->settingService->getSetting($field);
        if ($oldFile) {
            $this->deleteFile($oldFile);
        }

        // ایجاد نام یکتا برای فایل
        $extension = $file->getClientOriginalExtension();
        $fileName = $field . '_' . Str::random(10) . '.' . $extension;

        // آپلود فایل به مسیر public/images
        $publicPath = public_path('images');
        if (!file_exists($publicPath)) {
            mkdir($publicPath, 0777, true);
        }

        // ذخیره فایل
        $file->move($publicPath, $fileName);

        // لاگ برای عیب‌یابی
        \Log::info('SiteSetting Upload', [
            'field' => $field,
            'fileName' => $fileName,
            'path' => '/images/' . $fileName
        ]);

        // برگرداندن مسیر نسبی برای ذخیره در دیتابیس
        return '/images/' . $fileName;
    }

    /**
     * حذف فایل
     *
     * @param string $filePath
     * @return void
     */
    protected function deleteFile($filePath)
    {
        if (!$filePath) {
            return;
        }

        try {
            // حذف از مسیر public
            $publicPath = public_path(ltrim($filePath, '/'));
            if (file_exists($publicPath)) {
                unlink($publicPath);
            }

            // لاگ برای عیب‌یابی
            \Log::info('SiteSetting Delete', [
                'filePath' => $filePath,
                'publicPath' => $publicPath,
                'exists' => file_exists($publicPath)
            ]);
        } catch (\Exception $e) {
            \Log::error('Error deleting file: ' . $e->getMessage());
        }
    }

    /**
     * به‌روزرسانی تنظیمات
     *
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(Request $request)
    {
        return $this->store($request);
    }

    /**
     * پاک کردن کش تنظیمات
     *
     * @return \Illuminate\Http\RedirectResponse
     */
    public function clearCache()
    {
        $this->settingService->clearCache();
        return redirect()->route('admin.settings.index')
            ->with('success', 'کش تنظیمات با موفقیت پاک شد.');
    }

    /**
     * تغییر وضعیت حالت تعمیر و نگهداری
     *
     * @return \Illuminate\Http\RedirectResponse
     */
    public function toggleMaintenanceMode()
    {
        $isMaintenance = $this->settingService->toggleMaintenanceMode(app()->getLocale());
        $message = $isMaintenance ? 'حالت تعمیر و نگهداری فعال شد.' : 'حالت تعمیر و نگهداری غیرفعال شد.';
        
        return redirect()->route('admin.settings.index')
            ->with('success', $message);
    }
} 