<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\AdDisplayZone;
use App\Models\AdPackage;
use App\Models\LocationAdPackage;
use App\Services\AdService;
use Illuminate\Http\Request;

class AdPackageController extends Controller
{
    protected $adService;

    public function __construct(AdService $adService)
    {
        $this->adService = $adService;
    }

    /**
     * نمایش فهرست پکیج‌های تبلیغاتی
     */
    public function index()
    {
        $packages = AdPackage::orderBy('priority')->get();

        return view('admin.packages.index', compact('packages'));
    }

    /**
     * نمایش فرم ایجاد پکیج جدید
     */
    public function create()
    {
        return view('admin.packages.create');
    }

    /**
     * ذخیره پکیج جدید
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'package_key' => 'required|unique:ad_packages',
            'name' => 'required|string|max:100',
            'description' => 'nullable|string',
            'price' => 'required|numeric|min:0',
            'duration_days' => 'required|integer|min:1',
            'badge_text' => 'nullable|string|max:50',
            'badge_color' => 'nullable|string|max:20',
            'card_design' => 'required|string|max:50',
            'card_border_color' => 'nullable|string|max:20',
            'card_background_color' => 'nullable|string|max:20',
            'card_text_color' => 'nullable|string|max:20',
            'priority' => 'required|integer|min:1',
            'is_active' => 'boolean',
        ]);

        $package = AdPackage::create($validated);

        return redirect()->route('admin.packages.index')
            ->with('success', 'پکیج تبلیغاتی با موفقیت ایجاد شد.');
    }

    /**
     * نمایش فرم ویرایش پکیج
     */
    public function edit($id)
    {
        $package = AdPackage::findOrFail($id);

        return view('admin.packages.edit', compact('package'));
    }

    /**
     * بروزرسانی پکیج
     */
    public function update(Request $request, $id)
    {
        $package = AdPackage::findOrFail($id);

        $validated = $request->validate([
            'package_key' => 'required|unique:ad_packages,package_key,'.$id,
            'name' => 'required|string|max:100',
            'description' => 'nullable|string',
            'price' => 'required|numeric|min:0',
            'duration_days' => 'required|integer|min:1',
            'badge_text' => 'nullable|string|max:50',
            'badge_color' => 'nullable|string|max:20',
            'card_design' => 'required|string|max:50',
            'card_border_color' => 'nullable|string|max:20',
            'card_background_color' => 'nullable|string|max:20',
            'card_text_color' => 'nullable|string|max:20',
            'priority' => 'required|integer|min:1',
            'is_active' => 'boolean',
        ]);

        $package->update($validated);

        return redirect()->route('admin.packages.index')
            ->with('success', 'پکیج تبلیغاتی با موفقیت بروزرسانی شد.');
    }

    /**
     * مدیریت تبلیغات مکان‌ها
     */
    public function locationAds()
    {
        $locationAds = LocationAdPackage::with(['location', 'package'])
            ->orderBy('end_date', 'desc')
            ->paginate(20);

        return view('admin.ads.location-ads', compact('locationAds'));
    }

    /**
     * نمایش فرم اختصاص پکیج به مکان
     */
    public function assignPackage()
    {
        $locations = \App\Models\Location::where('is_active', 1)->get();
        $packages = AdPackage::where('is_active', 1)->orderBy('priority')->get();

        return view('admin.ads.assign-package', compact('locations', 'packages'));
    }

    /**
     * ذخیره اختصاص پکیج به مکان
     */
    public function storeAssignment(Request $request)
    {
        $validated = $request->validate([
            'location_id' => 'required|exists:locations,id',
            'package_id' => 'required|exists:ad_packages,id',
            'start_date' => 'required|date',
            'end_date' => 'nullable|date|after:start_date',
            'display_priority' => 'nullable|integer|min:1',
            'geographic_scope' => 'nullable|string',
            'geo_id' => 'nullable|integer',
            'category_id' => 'nullable|exists:categories,id',
            'custom_badge_text' => 'nullable|string|max:50',
            'custom_badge_color' => 'nullable|string|max:20',
        ]);

        $locationPackage = $this->adService->purchasePackage(
            $validated['location_id'],
            $validated['package_id'],
            $validated
        );

        // فرض می‌کنیم پرداخت انجام شده است (در محیط واقعی به درگاه پرداخت هدایت می‌شود)
        $this->adService->completePayment($locationPackage->id, 'ADMIN_MANUAL_'.time());

        return redirect()->route('admin.ads.location-ads')
            ->with('success', 'پکیج تبلیغاتی با موفقیت به مکان اختصاص داده شد.');
    }

    /**
     * مدیریت ناحیه‌های نمایش تبلیغات
     */
    public function displayZones()
    {
        $zones = AdDisplayZone::orderBy('display_order')->get();

        return view('admin.ads.display-zones', compact('zones'));
    }

    /**
     * بروزرسانی ناحیه نمایش
     */
    public function updateDisplayZone(Request $request, $id)
    {
        $zone = AdDisplayZone::findOrFail($id);

        $validated = $request->validate([
            'zone_name' => 'required|string|max:100',
            'description' => 'nullable|string',
            'max_desktop_cards' => 'required|integer|min:1|max:10',
            'max_mobile_cards' => 'required|integer|min:1|max:6',
            'card_layout' => 'required|string|in:grid,carousel,list',
            'is_active' => 'boolean',
            'display_order' => 'required|integer|min:1',
            'allowed_packages' => 'nullable|array',
            'allowed_packages.*' => 'string|exists:ad_packages,package_key',
        ]);

        $zone->update($validated);

        return redirect()->route('admin.ads.display-zones')
            ->with('success', 'ناحیه نمایش با موفقیت بروزرسانی شد.');
    }
}
