<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use App\Models\Category;
use App\Models\City;
use App\Models\Country;
use App\Models\County;
use App\Models\District;
use App\Models\Location;
use App\Models\Province;
use App\Models\Village;
use App\Models\Zone;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class LocationSubmissionController extends Controller
{
    public function index(Request $request)
    {
        $user = Auth::user();

        // Get filter parameters
        $status = $request->get('status', 'all');
        $category = $request->get('category');
        $search = $request->get('search');
        $sort = $request->get('sort', 'newest');

        try {
            // Build query
            $query = Location::where('user_id', $user->id)
                ->with(['categoryRelation', 'province', 'city']);

            // Apply filters
            if ($status !== 'all') {
                if ($status === 'approved') {
                    $query->where('is_active', 1);
                } elseif ($status === 'pending') {
                    $query->where('is_active', 0);
                } elseif ($status === 'rejected') {
                    $query->where('is_active', -1);
                }
            }

            if ($category) {
                $query->where('category_id', $category);
            }

            if ($search) {
                $query->where(function($q) use ($search) {
                    $q->where('title', 'like', "%{$search}%")
                      ->orWhere('description', 'like', "%{$search}%")
                      ->orWhere('address', 'like', "%{$search}%");
                });
            }

            // Apply sorting
            switch ($sort) {
                case 'oldest':
                    $query->orderBy('created_at', 'asc');
                    break;
                case 'title':
                    $query->orderBy('title', 'asc');
                    break;
                case 'status':
                    $query->orderBy('is_active', 'desc');
                    break;
                default:
                    $query->orderBy('created_at', 'desc');
            }

            $locations = $query->paginate(12)->withQueryString();

        } catch (\Exception $e) {
            $locations = collect([]);
        }

        // Get statistics
        $stats = $this->getLocationStats($user);

        // Get categories for filter
        $categories = Category::orderBy('category_fa')->get();

        // Check if it's an AJAX request for infinite scroll
        if ($request->ajax()) {
            return response()->json([
                'html' => view('user.locations.partials.location-grid', compact('locations'))->render(),
                'hasMorePages' => $locations->hasMorePages(),
                'nextPageUrl' => $locations->nextPageUrl()
            ]);
        }

        return view('user.locations.index', compact('locations', 'stats', 'categories', 'status', 'category', 'search', 'sort'));
    }

    /**
     * Get location statistics for user
     */
    private function getLocationStats($user)
    {
        try {
            $total = Location::where('user_id', $user->id)->count();
            $approved = Location::where('user_id', $user->id)->where('is_active', 1)->count();
            $pending = Location::where('user_id', $user->id)->where('is_active', 0)->count();
            $rejected = Location::where('user_id', $user->id)->where('is_active', -1)->count();

            $totalViews = Location::where('user_id', $user->id)->sum('views_count') ?? 0;

            // Get total ratings for user's locations
            $totalRatings = \DB::table('ratings')
                ->join('locations', 'ratings.location_id', '=', 'locations.id')
                ->where('locations.user_id', $user->id)
                ->count();

            $avgRating = \DB::table('ratings')
                ->join('locations', 'ratings.location_id', '=', 'locations.id')
                ->where('locations.user_id', $user->id)
                ->avg('ratings.rating') ?? 0;

            return [
                'total' => $total,
                'approved' => $approved,
                'pending' => $pending,
                'rejected' => $rejected,
                'total_views' => $totalViews,
                'total_ratings' => $totalRatings,
                'avg_rating' => round($avgRating, 1)
            ];
        } catch (\Exception $e) {
            return [
                'total' => 0,
                'approved' => 0,
                'pending' => 0,
                'rejected' => 0,
                'total_views' => 0,
                'total_ratings' => 0,
                'avg_rating' => 0
            ];
        }
    }

    public function create()
    {
        // دریافت کلیه دسته‌بندی‌ها - والد و فرزند - به صورت مسطح
        $parentCategories = Category::where('parent_id', null)
            ->with('children')
            ->orderBy('sort_order')
            ->get();

        $childCategories = collect([]);
        foreach ($parentCategories as $parent) {
            if ($parent->children && $parent->children->count() > 0) {
                $childCategories = $childCategories->concat($parent->children);
            }
        }

        // ادغام دسته‌بندی‌ها و ایجاد ساختار گروه‌بندی شده برای نمایش در منوی کشویی
        $categoriesForDropdown = collect([]);
        foreach ($parentCategories as $parent) {
            // اضافه کردن دسته‌بندی والد به عنوان گروه
            $categoriesForDropdown->push([
                'id' => $parent->id,
                'text' => $parent->getTranslatedName(app()->getLocale()),
                'is_parent' => true,
            ]);

            // اضافه کردن دسته‌بندی‌های فرزند
            if ($parent->children && $parent->children->count() > 0) {
                foreach ($parent->children as $child) {
                    $categoriesForDropdown->push([
                        'id' => $child->id,
                        'text' => '— '.$child->getTranslatedName(app()->getLocale()),
                        'parent_id' => $parent->id,
                        'is_parent' => false,
                    ]);
                }
            }
        }

        $countries = Country::where('is_active', 1)->get();
        $provinces = Province::where('is_active', 1)->get();
        $counties = County::all();
        $cities = City::all();
        $districts = District::all();
        $villages = Village::all();
        $zones = Zone::all();

        return view('user.locations.create', compact(
            'parentCategories',
            'childCategories',
            'categoriesForDropdown',
            'countries',
            'provinces',
            'counties',
            'cities',
            'districts',
            'villages',
            'zones'
        ));
    }

    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'category_id' => 'required|exists:categories,id',
            'description' => 'required|string',
            'country_id' => 'required|exists:state1_countries,id',
            'province_id' => 'required|exists:state2_provinces,id',
            'county_id' => 'nullable|exists:state3_counties,id',
            'city_id' => 'nullable|exists:state5_cities,id',
            'address' => 'required|string',
            'phone' => 'nullable|string|max:50',
            'lat' => 'nullable|numeric',
            'lng' => 'nullable|numeric',
        ]);

        // Get category information
        $category = Category::find($request->category_id);

        $location = new Location;
        $location->title = $request->title;
        $location->category_id = $request->category_id;
        $location->category = $category->category_key; // Using category key from database
        $location->category_fa = $category->category_fa;
        $location->description = $request->description;
        $location->slogan = $request->slogan ?? '';
        $location->country_id = $request->country_id;
        $location->province_id = $request->province_id;
        $location->county_id = $request->county_id;
        $location->city_id = $request->city_id;
        $location->address = $request->address;
        $location->phone = $request->phone;
        $location->lat = $request->lat;
        $location->lng = $request->lng;
        $location->is_active = 0; // Pending approval
        $location->title_id = uniqid('loc_');
        $location->user_id = Auth::id();
        $location->emoji = $category->emoji ?? '📍';

        $location->save();

        return redirect()->route('user.locations.index')
            ->with('success', 'درخواست ثبت مکان جدید با موفقیت ارسال شد و پس از بررسی منتشر خواهد شد.');
    }

    public function show($id)
    {
        $user = Auth::user();
        $location = Location::where('id', $id)
            ->where('user_id', $user->id)
            ->firstOrFail();

        return view('user.locations.show', compact('location'));
    }

    public function edit($id)
    {
        $user = Auth::user();
        $location = Location::where('id', $id)
            ->where('user_id', $user->id)
            ->firstOrFail();

        $categories = Category::where('parent_id', null)
            ->with('children')
            ->orderBy('sort_order')
            ->get();

        $countries = Country::where('is_active', 1)->get();
        $provinces = Province::where('country_id', $location->country_id)->get();

        return view('user.locations.edit', compact('location', 'categories', 'countries', 'provinces'));
    }

    public function update(Request $request, $id)
    {
        $user = Auth::user();
        $location = Location::where('id', $id)
            ->where('user_id', $user->id)
            ->firstOrFail();

        // Only allow edits if the location is not already approved
        if ($location->is_active) {
            return redirect()->route('user.locations.index')
                ->with('error', 'مکان‌های تایید شده قابل ویرایش نیستند.');
        }

        $request->validate([
            'title' => 'required|string|max:255',
            'category_id' => 'required|exists:categories,id',
            'description' => 'required|string',
            'country_id' => 'required|exists:state1_countries,id',
            'province_id' => 'required|exists:state2_provinces,id',
            'county_id' => 'nullable|exists:state3_counties,id',
            'city_id' => 'nullable|exists:state5_cities,id',
            'address' => 'required|string',
            'phone' => 'nullable|string|max:50',
            'lat' => 'nullable|numeric',
            'lng' => 'nullable|numeric',
        ]);

        // Get category information
        $category = Category::find($request->category_id);

        $location->title = $request->title;
        $location->category_id = $request->category_id;
        $location->category = $category->category_key;
        $location->category_fa = $category->category_fa;
        $location->description = $request->description;
        $location->slogan = $request->slogan ?? '';
        $location->country_id = $request->country_id;
        $location->province_id = $request->province_id;
        $location->county_id = $request->county_id;
        $location->city_id = $request->city_id;
        $location->address = $request->address;
        $location->phone = $request->phone;
        $location->lat = $request->lat;
        $location->lng = $request->lng;
        $location->emoji = $category->emoji ?? '📍';

        $location->save();

        return redirect()->route('user.locations.index')
            ->with('success', 'درخواست ثبت مکان با موفقیت به‌روزرسانی شد.');
    }

    public function destroy($id)
    {
        $user = Auth::user();
        $location = Location::where('id', $id)
            ->where('user_id', $user->id)
            ->firstOrFail();

        $location->delete();

        return redirect()->route('user.locations.index')
            ->with('success', 'مکان با موفقیت حذف شد.');
    }
}
