<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\ButtonSetting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;

class UserMenuController extends Controller
{
    /**
     * Display user menu management page.
     */
    public function index(Request $request)
    {
        $languageCode = $request->get('lang', 'fa');
        
        // Get buttons by type
        $guestButtons = ButtonSetting::forLanguage($languageCode)
            ->where('show_for_guests', 1)
            ->where('show_for_users', 0)
            ->orderBy('sort_order')
            ->get();
            
        $userButtons = ButtonSetting::forLanguage($languageCode)
            ->where('show_for_guests', 0)
            ->where('show_for_users', 1)
            ->orderBy('sort_order')
            ->get();
            
        $sharedButtons = ButtonSetting::forLanguage($languageCode)
            ->where('show_for_guests', 1)
            ->where('show_for_users', 1)
            ->orderBy('sort_order')
            ->get();

        return view('admin.user-menu-management.index', compact(
            'languageCode',
            'guestButtons',
            'userButtons', 
            'sharedButtons'
        ));
    }

    /**
     * Update buttons based on type.
     */
    public function updateButtons(Request $request)
    {
        $validated = $request->validate([
            'language_code' => 'required|string|max:5',
            'button_type' => 'required|in:guest,user,shared',
            'buttons' => 'required|array',
            'buttons.*.button_key' => 'required|string|max:50',
            'buttons.*.button_text' => 'required|string|max:100',
            'buttons.*.button_url' => 'nullable|string|max:255',
            'buttons.*.button_icon' => 'nullable|string|max:50',
            'buttons.*.sort_order' => 'required|integer|min:0',
            'buttons.*.is_visible' => 'required|in:0,1',
            'buttons.*.show_for_guests' => 'required|in:0,1',
            'buttons.*.show_for_users' => 'required|in:0,1',
        ]);

        $languageCode = $validated['language_code'];
        $buttonType = $validated['button_type'];

        foreach ($validated['buttons'] as $buttonData) {
            ButtonSetting::updateButton(
                $buttonData['button_key'],
                array_merge($buttonData, ['language_code' => $languageCode]),
                $languageCode
            );
        }

        // Clear cache
        Cache::forget('button_settings_' . $languageCode);

        $message = match($buttonType) {
            'guest' => 'دکمه‌های مهمان با موفقیت به‌روزرسانی شدند.',
            'user' => 'دکمه‌های کاربر با موفقیت به‌روزرسانی شدند.',
            'shared' => 'دکمه‌های مشترک با موفقیت به‌روزرسانی شدند.',
        };

        return redirect()->route('admin.user-menu.index', ['lang' => $languageCode])
            ->with('success', $message);
    }

    /**
     * Add new button.
     */
    public function addButton(Request $request)
    {
        $validated = $request->validate([
            'button_key' => 'required|string|max:50|unique:button_settings,button_key',
            'button_text' => 'required|string|max:100',
            'button_url' => 'nullable|string|max:255',
            'button_icon' => 'nullable|string|max:50',
            'sort_order' => 'required|integer|min:0',
            'button_type' => 'required|in:guest,user,shared',
            'language_code' => 'required|string|max:5',
        ]);

        // Set visibility based on button type
        $buttonData = [
            'button_key' => $validated['button_key'],
            'button_text' => $validated['button_text'],
            'button_url' => $validated['button_url'],
            'button_icon' => $validated['button_icon'],
            'sort_order' => $validated['sort_order'],
            'language_code' => $validated['language_code'],
            'is_visible' => 1,
        ];

        switch ($validated['button_type']) {
            case 'guest':
                $buttonData['show_for_guests'] = 1;
                $buttonData['show_for_users'] = 0;
                break;
            case 'user':
                $buttonData['show_for_guests'] = 0;
                $buttonData['show_for_users'] = 1;
                break;
            case 'shared':
                $buttonData['show_for_guests'] = 1;
                $buttonData['show_for_users'] = 1;
                break;
        }

        ButtonSetting::create($buttonData);

        // Clear cache
        Cache::forget('button_settings_' . $validated['language_code']);

        return redirect()->route('admin.user-menu.index', ['lang' => $validated['language_code']])
            ->with('success', 'دکمه جدید با موفقیت اضافه شد.');
    }

    /**
     * Toggle button visibility.
     */
    public function toggleButtonVisibility(ButtonSetting $button)
    {
        $button->update(['is_visible' => !$button->is_visible]);

        // Clear cache
        Cache::forget('button_settings_' . $button->language_code);

        return response()->json([
            'success' => true,
            'is_visible' => $button->is_visible,
            'message' => $button->is_visible ? 'دکمه نمایش داده می‌شود.' : 'دکمه مخفی شد.'
        ]);
    }

    /**
     * Reset to default buttons.
     */
    public function resetToDefault(Request $request)
    {
        $languageCode = $request->get('lang', 'fa');

        // Validate language code
        $supportedLanguages = ['fa', 'en', 'ar'];
        if (!in_array($languageCode, $supportedLanguages)) {
            $languageCode = 'fa';
        }

        // Delete existing buttons
        ButtonSetting::forLanguage($languageCode)->delete();

        // Create default buttons based on language
        $defaultButtons = $this->getDefaultButtonsForLanguage($languageCode);

        foreach ($defaultButtons as $buttonData) {
            ButtonSetting::create(array_merge($buttonData, [
                'language_code' => $languageCode,
                'is_visible' => 1,
                'button_class' => 'btn-primary'
            ]));
        }

        // Clear cache
        Cache::forget('button_settings_' . $languageCode);

        return redirect()->route('admin.user-menu.index', ['lang' => $languageCode])
            ->with('success', 'دکمه‌های کاربری به حالت پیش‌فرض بازگردانده شدند.');
    }

    /**
     * Get default buttons for specific language
     */
    private function getDefaultButtonsForLanguage($languageCode)
    {
        $buttons = [
            'fa' => [
                'login' => 'ورود',
                'register' => 'ثبت نام',
                'dashboard' => 'داشبورد',
                'profile' => 'پروفایل کاربری',
                'my_locations' => 'مکان‌های من',
                'favorites' => 'علاقه‌مندی‌ها',
                'settings' => 'تنظیمات',
                'logout' => 'خروج',
                'add_location' => 'ثبت مکان جدید',
            ],
            'en' => [
                'login' => 'Login',
                'register' => 'Register',
                'dashboard' => 'Dashboard',
                'profile' => 'User Profile',
                'my_locations' => 'My Locations',
                'favorites' => 'Favorites',
                'settings' => 'Settings',
                'logout' => 'Logout',
                'add_location' => 'Add New Location',
            ],
            'ar' => [
                'login' => 'تسجيل الدخول',
                'register' => 'إنشاء حساب',
                'dashboard' => 'لوحة التحكم',
                'profile' => 'الملف الشخصي',
                'my_locations' => 'مواقعي',
                'favorites' => 'المفضلة',
                'settings' => 'الإعدادات',
                'logout' => 'تسجيل الخروج',
                'add_location' => 'إضافة موقع جديد',
            ],
        ];

        $langButtons = $buttons[$languageCode] ?? $buttons['fa'];

        return [
            // Guest buttons
            ['button_key' => 'login', 'button_text' => $langButtons['login'], 'button_url' => '/login', 'button_icon' => 'fas fa-sign-in-alt', 'button_class' => 'btn-auth', 'is_visible' => 1, 'show_for_guests' => 1, 'show_for_users' => 0, 'sort_order' => 1],
            ['button_key' => 'register', 'button_text' => $langButtons['register'], 'button_url' => '/register', 'button_icon' => 'fas fa-user-plus', 'button_class' => 'btn-auth', 'is_visible' => 1, 'show_for_guests' => 1, 'show_for_users' => 0, 'sort_order' => 2],

            // User buttons
            ['button_key' => 'dashboard', 'button_text' => $langButtons['dashboard'], 'button_url' => '/user/dashboard', 'button_icon' => 'fas fa-tachometer-alt', 'button_class' => 'dropdown-item', 'is_visible' => 1, 'show_for_guests' => 0, 'show_for_users' => 1, 'sort_order' => 1],
            ['button_key' => 'profile', 'button_text' => $langButtons['profile'], 'button_url' => '/user/profile', 'button_icon' => 'fas fa-user-edit', 'button_class' => 'dropdown-item', 'is_visible' => 1, 'show_for_guests' => 0, 'show_for_users' => 1, 'sort_order' => 2],
            ['button_key' => 'my_locations', 'button_text' => $langButtons['my_locations'], 'button_url' => '/user/locations', 'button_icon' => 'fas fa-map-marker-alt', 'button_class' => 'dropdown-item', 'is_visible' => 1, 'show_for_guests' => 0, 'show_for_users' => 1, 'sort_order' => 3],
            ['button_key' => 'favorites', 'button_text' => $langButtons['favorites'], 'button_url' => '/user/favorites', 'button_icon' => 'fas fa-heart', 'button_class' => 'dropdown-item', 'is_visible' => 1, 'show_for_guests' => 0, 'show_for_users' => 1, 'sort_order' => 4],
            ['button_key' => 'settings', 'button_text' => $langButtons['settings'], 'button_url' => '/user/settings', 'button_icon' => 'fas fa-cog', 'button_class' => 'dropdown-item', 'is_visible' => 1, 'show_for_guests' => 0, 'show_for_users' => 1, 'sort_order' => 5],
            ['button_key' => 'logout', 'button_text' => $langButtons['logout'], 'button_url' => '/user/logout', 'button_icon' => 'fas fa-sign-out-alt', 'button_class' => 'dropdown-item', 'is_visible' => 1, 'show_for_guests' => 0, 'show_for_users' => 1, 'sort_order' => 6],

            // Shared buttons
            ['button_key' => 'add_location', 'button_text' => $langButtons['add_location'], 'button_url' => '/user/locations/create', 'button_icon' => 'fas fa-plus-circle', 'button_class' => 'btn-add-place', 'is_visible' => 1, 'show_for_guests' => 1, 'show_for_users' => 1, 'sort_order' => 10],
        ];
    }
}
