<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\SecuritySetting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;

class SecuritySettingController extends Controller
{
    public function index()
    {
        $groups = [
            'file' => 'تنظیمات فایل',
            'session' => 'تنظیمات نشست',
            'xss' => 'تنظیمات XSS و CSRF',
            'brute_force' => 'تنظیمات محافظت در برابر حملات',
            'log' => 'تنظیمات لاگ',
        ];

        $settings = [];
        foreach ($groups as $group => $title) {
            $settings[$group] = SecuritySetting::where('group', $group)->get();
        }

        return view('admin.security-settings.index', compact('groups', 'settings'));
    }

    public function update(Request $request)
    {
        $validated = $request->validate([
            'settings' => 'required|array',
            'settings.*.group' => 'required|string',
            'settings.*.key' => 'required|string',
            'settings.*.value' => 'required',
            'settings.*.description' => 'nullable|string',
        ]);

        foreach ($validated['settings'] as $setting) {
            SecuritySetting::setValue(
                $setting['group'],
                $setting['key'],
                $setting['value'],
                $setting['description'] ?? null
            );
        }

        // پاک کردن کش تنظیمات
        Cache::tags(['security_settings'])->flush();

        return response()->json([
            'message' => 'تنظیمات با موفقیت بروزرسانی شد.',
            'status' => 'success',
        ]);
    }

    public function getSettings(string $group)
    {
        $settings = SecuritySetting::where('group', $group)->get();

        return response()->json($settings);
    }

    public function toggleSetting(Request $request)
    {
        $validated = $request->validate([
            'group' => 'required|string',
            'key' => 'required|string',
            'is_active' => 'required|boolean',
        ]);

        $setting = SecuritySetting::where('group', $validated['group'])
            ->where('key', $validated['key'])
            ->first();

        if ($setting) {
            $setting->update(['is_active' => $validated['is_active']]);
            Cache::tags(['security_settings'])->flush();
        }

        return response()->json([
            'message' => 'وضعیت تنظیم با موفقیت تغییر کرد.',
            'status' => 'success',
        ]);
    }

    /**
     * نمایش تاریخچه ورود کاربران
     *
     * @return \Illuminate\View\View
     */
    public function loginHistory()
    {
        $loginHistory = DB::table('admin_login_history')
            ->join('admins', 'admin_login_history.admin_id', '=', 'admins.id')
            ->select('admin_login_history.*', 'admins.name', 'admins.email')
            ->orderBy('admin_login_history.created_at', 'desc')
            ->paginate(20);

        return view('admin.security-settings.login-history', compact('loginHistory'));
    }

    /**
     * نمایش ماژول‌های فعال
     *
     * @return \Illuminate\View\View
     */
    public function activeModules()
    {
        $modules = [
            'booking' => [
                'name' => 'رزرواسیون',
                'description' => 'مدیریت سیستم رزرواسیون',
                'is_active' => true
            ],
            'payment' => [
                'name' => 'پرداخت',
                'description' => 'مدیریت سیستم پرداخت',
                'is_active' => true
            ],
            'gallery' => [
                'name' => 'گالری',
                'description' => 'مدیریت گالری تصاویر',
                'is_active' => true
            ],
            'comments' => [
                'name' => 'نظرات',
                'description' => 'مدیریت نظرات کاربران',
                'is_active' => true
            ],
            'map' => [
                'name' => 'نقشه',
                'description' => 'نمایش نقشه با Leaflet',
                'is_active' => true
            ],
            'opening_hours' => [
                'name' => 'ساعات کاری',
                'description' => 'مدیریت ساعات کاری',
                'is_active' => true
            ]
        ];

        return view('admin.modules.active', compact('modules'));
    }

    /**
     * نصب ماژول جدید
     *
     * @return \Illuminate\View\View
     */
    public function installModule()
    {
        $availableModules = [
            'booking' => [
                'name' => 'رزرواسیون',
                'description' => 'مدیریت سیستم رزرواسیون',
                'version' => '1.0.0',
                'is_installed' => false
            ],
            'payment' => [
                'name' => 'پرداخت',
                'description' => 'مدیریت سیستم پرداخت',
                'version' => '1.0.0',
                'is_installed' => false
            ],
            'gallery' => [
                'name' => 'گالری',
                'description' => 'مدیریت گالری تصاویر',
                'version' => '1.0.0',
                'is_installed' => false
            ],
            'comments' => [
                'name' => 'نظرات',
                'description' => 'مدیریت نظرات کاربران',
                'version' => '1.0.0',
                'is_installed' => false
            ],
            'map' => [
                'name' => 'نقشه',
                'description' => 'نمایش نقشه با Leaflet',
                'version' => '1.0.0',
                'is_installed' => false
            ],
            'opening_hours' => [
                'name' => 'ساعات کاری',
                'description' => 'مدیریت ساعات کاری',
                'version' => '1.0.0',
                'is_installed' => false
            ]
        ];

        return view('admin.modules.install', compact('availableModules'));
    }

    /**
     * ایجاد نسخه پشتیبان از دیتابیس
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function createBackup()
    {
        try {
            $backup = new \App\Services\BackupService();
            $result = $backup->create();

            if ($result['success']) {
                return response()->json([
                    'message' => 'نسخه پشتیبان با موفقیت ایجاد شد.',
                    'status' => 'success',
                    'file' => $result['file']
                ]);
            }

            return response()->json([
                'message' => 'خطا در ایجاد نسخه پشتیبان.',
                'status' => 'error'
            ], 500);

        } catch (\Exception $e) {
            return response()->json([
                'message' => 'خطا در ایجاد نسخه پشتیبان: ' . $e->getMessage(),
                'status' => 'error'
            ], 500);
        }
    }
}
