<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Rating;
use App\Models\RatingLocationComment;
use App\Models\RatingStatistics;
use App\Models\Location;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class RatingController extends Controller
{
    /**
     * Display a listing of the ratings with advanced filters and statistics.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        // Get filter parameters
        $status = $request->get('status', 'all');
        $category = $request->get('category');
        $rating_filter = $request->get('rating_filter');
        $date_from = $request->get('date_from');
        $date_to = $request->get('date_to');
        $search = $request->get('search');
        $per_page = $request->get('per_page', 15);

        // Build query
        $query = Rating::with([
            'location' => function($q) {
                $q->select('title_id', 'title', 'category');
            },
            'categoryRelation' => function($q) {
                $q->select('id', 'category_key', 'category_fa', 'category_en', 'category_ar');
            }
        ]);

        // Apply filters
        if ($status !== 'all') {
            $query->where('status', $status);
        }

        if ($category) {
            $query->where('category', $category);
        }

        if ($rating_filter) {
            $query->where('rating', $rating_filter);
        }

        if ($date_from) {
            $query->whereDate('timestamp', '>=', $date_from);
        }

        if ($date_to) {
            $query->whereDate('timestamp', '<=', $date_to);
        }

        if ($search) {
            $query->where(function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%")
                  ->orWhere('hotel_name', 'like', "%{$search}%")
                  ->orWhere('comment', 'like', "%{$search}%")
                  ->orWhere('positive_comment', 'like', "%{$search}%")
                  ->orWhere('negative_comment', 'like', "%{$search}%");
            });
        }

        $ratings = $query->latest('timestamp')->paginate($per_page);

        // Get statistics for dashboard cards
        $stats = $this->getRatingStatistics();

        // Debug: نمایش آمار در log
        \Log::info('Rating Statistics:', $stats);

        // Get categories for filter
        $categories = Rating::select('category')
            ->whereNotNull('category')
            ->distinct()
            ->pluck('category');

        return view('admin.ratings.index', compact(
            'ratings',
            'stats',
            'categories',
            'status',
            'category',
            'rating_filter',
            'date_from',
            'date_to',
            'search',
            'per_page'
        ));
    }

    /**
     * Approve a rating
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function approve($id)
    {
        try {
            $rating = Rating::findOrFail($id);
            $rating->status = 'approved';
            $rating->save();

            return redirect()->route('admin.ratings.index')
                ->with('success', 'امتیاز با موفقیت تایید شد.');
        } catch (\Exception $e) {
            return redirect()->route('admin.ratings.index')
                ->with('error', 'خطایی در تایید امتیاز رخ داد.');
        }
    }

    /**
     * Reject a rating
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function reject($id)
    {
        try {
            $rating = Rating::findOrFail($id);
            $rating->status = 'rejected';
            $rating->save();

            return redirect()->route('admin.ratings.index')
                ->with('success', 'امتیاز با موفقیت رد شد.');
        } catch (\Exception $e) {
            return redirect()->route('admin.ratings.index')
                ->with('error', 'خطایی در رد امتیاز رخ داد.');
        }
    }

    /**
     * Delete a rating
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function delete($id)
    {
        try {
            $rating = Rating::findOrFail($id);
            $rating->delete();

            return redirect()->route('admin.ratings.index')
                ->with('success', 'امتیاز با موفقیت حذف شد.');
        } catch (\Exception $e) {
            return redirect()->route('admin.ratings.index')
                ->with('error', 'خطایی در حذف امتیاز رخ داد.');
        }
    }

    /**
     * Show modern dashboard with comprehensive analytics
     *
     * @return \Illuminate\Http\Response
     */
    public function dashboard()
    {
        // Basic Statistics
        $totalRatings = Rating::count();
        $pendingRatings = Rating::where('status', 'pending')->count();
        $approvedRatings = Rating::where('status', 'approved')->count();
        $rejectedRatings = Rating::where('status', 'rejected')->count();
        $averageRating = Rating::where('status', 'approved')->avg('rating') ?? 0;

        // Growth Metrics (comparing with previous period)
        $thirtyDaysAgo = Carbon::now()->subDays(30);
        $sixtyDaysAgo = Carbon::now()->subDays(60);

        $currentPeriodRatings = Rating::where('timestamp', '>=', $thirtyDaysAgo)->count();
        $previousPeriodRatings = Rating::where('timestamp', '>=', $sixtyDaysAgo)
            ->where('timestamp', '<', $thirtyDaysAgo)->count();

        $ratingsGrowth = $previousPeriodRatings > 0
            ? (($currentPeriodRatings - $previousPeriodRatings) / $previousPeriodRatings) * 100
            : 0;

        // Comments Statistics
        $totalComments = RatingLocationComment::count();
        $pendingComments = RatingLocationComment::where('status', 'pending')->count();
        $approvedComments = RatingLocationComment::where('status', 'approved')->count();

        // Rating Distribution for Chart
        $ratingDistribution = [];
        for ($i = 1; $i <= 5; $i++) {
            $ratingDistribution[$i] = Rating::where('status', 'approved')
                ->where('rating', $i)->count();
        }

        // Monthly Trends (last 6 months)
        $monthlyTrends = [];
        for ($i = 5; $i >= 0; $i--) {
            $month = Carbon::now()->subMonths($i);
            $monthlyTrends[] = [
                'month' => $month->format('M Y'),
                'month_short' => $month->format('M'),
                'ratings_count' => Rating::whereYear('timestamp', $month->year)
                    ->whereMonth('timestamp', $month->month)
                    ->where('status', 'approved')->count(),
                'avg_rating' => Rating::whereYear('timestamp', $month->year)
                    ->whereMonth('timestamp', $month->month)
                    ->where('status', 'approved')->avg('rating') ?? 0
            ];
        }

        // Category Performance
        $categoryPerformance = Rating::select(
                'category',
                DB::raw('COUNT(*) as total_ratings'),
                DB::raw('AVG(rating) as avg_rating')
            )
            ->whereNotNull('category')
            ->where('category', '!=', '')
            ->where('status', 'approved')
            ->groupBy('category')
            ->orderBy('total_ratings', 'desc')
            ->limit(8)
            ->get();

        // Recent Activities
        $recentRatings = Rating::with('location')
            ->where('timestamp', '>=', $thirtyDaysAgo)
            ->where('status', 'approved')
            ->orderBy('timestamp', 'desc')
            ->limit(8)
            ->get();

        // Top Rated Locations
        $topLocations = RatingStatistics::with('location')
            ->where('total_ratings', '>=', 3)
            ->orderBy('average_rating', 'desc')
            ->orderBy('total_ratings', 'desc')
            ->limit(10)
            ->get();

        // User Engagement Metrics
        $userEngagement = [
            'unique_reviewers' => Rating::distinct('email')->count('email'),
            'repeat_reviewers' => Rating::select('email')
                ->groupBy('email')
                ->havingRaw('COUNT(*) > 1')
                ->count(),
            'avg_reviews_per_user' => $totalRatings > 0
                ? round($totalRatings / Rating::distinct('email')->count('email'), 1)
                : 0,
            'helpful_votes_total' => Rating::sum('helpful_count') ?? 0
        ];

        // Response Time Analytics
        $avgResponseTime = Rating::where('status', '!=', 'pending')
            ->whereNotNull('updated_at')
            ->selectRaw('AVG(TIMESTAMPDIFF(HOUR, timestamp, updated_at)) as avg_hours')
            ->value('avg_hours') ?? 0;

        return view('admin.ratings.dashboard', compact(
            'totalRatings',
            'pendingRatings',
            'approvedRatings',
            'rejectedRatings',
            'averageRating',
            'ratingsGrowth',
            'totalComments',
            'pendingComments',
            'approvedComments',
            'ratingDistribution',
            'monthlyTrends',
            'categoryPerformance',
            'recentRatings',
            'topLocations',
            'userEngagement',
            'avgResponseTime'
        ));
    }

    /**
     * نمایش لیست امتیازهای در انتظار تایید
     *
     * @return \Illuminate\Http\Response
     */
    public function pending()
    {
        // دریافت امتیازهای در انتظار تایید
        $ratings = Rating::where('status', 'pending')->latest('timestamp')->paginate(15);

        // دریافت آمار برای نمایش در کارت‌ها
        $stats = $this->getRatingStatistics();

        // دریافت دسته‌بندی‌ها برای فیلتر
        $categories = Rating::select('category')
            ->whereNotNull('category')
            ->distinct()
            ->pluck('category');

        // تنظیم متغیرهای پیش‌فرض برای فیلترها
        $status = 'pending';
        $category = null;
        $rating_filter = null;
        $date_from = null;
        $date_to = null;
        $search = null;
        $per_page = 15;

        // اگر view مخصوص pending وجود داشت، همان را نمایش بده، در غیر این صورت از index استفاده کن
        if (view()->exists('admin.ratings.pending')) {
            return view('admin.ratings.pending', compact(
                'ratings',
                'stats',
                'categories',
                'status',
                'category',
                'rating_filter',
                'date_from',
                'date_to',
                'search',
                'per_page'
            ));
        } else {
            return view('admin.ratings.index', compact(
                'ratings',
                'stats',
                'categories',
                'status',
                'category',
                'rating_filter',
                'date_from',
                'date_to',
                'search',
                'per_page'
            ));
        }
    }

    /**
     * Get comprehensive rating statistics
     *
     * @return array
     */
    private function getRatingStatistics()
    {
        $stats = [];

        // Basic counts
        $stats['total_ratings'] = Rating::count();
        $stats['pending_ratings'] = Rating::where('status', 'pending')->count();
        $stats['approved_ratings'] = Rating::where('status', 'approved')->count();
        $stats['rejected_ratings'] = Rating::where('status', 'rejected')->count();

        // Average rating
        $stats['average_rating'] = Rating::where('status', 'approved')->avg('rating') ?? 0;

        // Comments statistics
        $stats['total_comments'] = RatingLocationComment::count();
        $stats['pending_comments'] = RatingLocationComment::where('status', 'pending')->count();
        $stats['approved_comments'] = RatingLocationComment::where('status', 'approved')->count();

        // Recent activity (last 30 days)
        $thirtyDaysAgo = Carbon::now()->subDays(30);
        $stats['recent_ratings'] = Rating::where('timestamp', '>=', $thirtyDaysAgo)->count();
        $stats['recent_comments'] = RatingLocationComment::where('created_at', '>=', $thirtyDaysAgo)->count();

        // Rating distribution
        $stats['rating_distribution'] = [];
        for ($i = 1; $i <= 5; $i++) {
            $count = Rating::where('status', 'approved')
                ->where('rating', $i)->count();
            $stats['rating_distribution'][$i] = $count;
        }

        // Debug: نمایش توزیع امتیازات
        \Log::info('Rating distribution:', $stats['rating_distribution']);

        // Top categories by rating count
        try {
            $topCategories = Rating::select('category', DB::raw('count(*) as count'))
                ->whereNotNull('category')
                ->where('category', '!=', '')
                ->groupBy('category')
                ->orderBy('count', 'desc')
                ->limit(5)
                ->get();

            $stats['top_categories'] = $topCategories;

            // Debug: نمایش دسته‌بندی‌های برتر
            \Log::info('Top categories:', $stats['top_categories']->toArray());
        } catch (\Exception $e) {
            \Log::error('Error getting top categories: ' . $e->getMessage());
            $stats['top_categories'] = collect();
        }

        // Monthly trends (last 12 months)
        $stats['monthly_trends'] = [];
        for ($i = 11; $i >= 0; $i--) {
            $month = Carbon::now()->subMonths($i);
            $stats['monthly_trends'][] = [
                'month' => $month->format('Y-m'),
                'month_name' => $month->format('F Y'),
                'ratings' => Rating::whereYear('timestamp', $month->year)
                    ->whereMonth('timestamp', $month->month)->count(),
                'comments' => RatingLocationComment::whereYear('created_at', $month->year)
                    ->whereMonth('created_at', $month->month)->count()
            ];
        }

        return $stats;
    }

    /**
     * Show rating settings page
     *
     * @return \Illuminate\Http\Response
     */
    public function settings()
    {
        return view('admin.ratings.settings');
    }

    /**
     * Update rating settings
     *
     * @param Request $request
     * @return \Illuminate\Http\Response
     */
    public function updateSettings(Request $request)
    {
        $request->validate([
            'auto_approve_ratings' => 'boolean',
            'auto_approve_comments' => 'boolean',
            'require_email_verification' => 'boolean',
            'allow_anonymous_ratings' => 'boolean',
            'min_rating_comment_length' => 'integer|min:0|max:1000',
            'max_rating_comment_length' => 'integer|min:100|max:5000',
            'enable_helpful_votes' => 'boolean',
            'enable_admin_replies' => 'boolean',
        ]);

        // Here you would typically save to a settings table or config
        // For now, we'll use session flash
        return redirect()->route('admin.ratings.settings')
            ->with('success', 'تنظیمات با موفقیت به‌روزرسانی شد.');
    }

    /**
     * Show detailed analytics
     *
     * @return \Illuminate\Http\Response
     */
    public function analytics()
    {
        $analytics = [];

        // Detailed rating analytics
        $analytics['rating_trends'] = $this->getRatingTrends();
        $analytics['category_performance'] = $this->getCategoryPerformance();
        $analytics['user_engagement'] = $this->getUserEngagement();
        $analytics['location_rankings'] = $this->getLocationRankings();

        return view('admin.ratings.analytics', compact('analytics'));
    }

    /**
     * Get rating trends data
     */
    private function getRatingTrends()
    {
        return Rating::select(
                DB::raw('DATE(timestamp) as date'),
                DB::raw('AVG(rating) as avg_rating'),
                DB::raw('COUNT(*) as count')
            )
            ->where('timestamp', '>=', Carbon::now()->subDays(30))
            ->where('status', 'approved')
            ->groupBy('date')
            ->orderBy('date')
            ->get();
    }

    /**
     * Get category performance data
     */
    private function getCategoryPerformance()
    {
        return Rating::select(
                'category',
                DB::raw('COUNT(*) as total_ratings'),
                DB::raw('AVG(rating) as avg_rating'),
                DB::raw('AVG(service_rating) as avg_service'),
                DB::raw('AVG(cleanliness_rating) as avg_cleanliness'),
                DB::raw('AVG(location_rating) as avg_location'),
                DB::raw('AVG(price_rating) as avg_price')
            )
            ->whereNotNull('category')
            ->where('status', 'approved')
            ->groupBy('category')
            ->orderBy('total_ratings', 'desc')
            ->get();
    }

    /**
     * Get user engagement data
     */
    private function getUserEngagement()
    {
        return [
            'repeat_reviewers' => Rating::select('email')
                ->groupBy('email')
                ->havingRaw('COUNT(*) > 1')
                ->count(),
            'verified_reviews' => Rating::where('is_verified', true)->count(),
            'helpful_votes_total' => Rating::sum('helpful_count'),
            'avg_helpful_ratio' => Rating::where('helpful_count', '>', 0)
                ->avg(DB::raw('helpful_count / (helpful_count + not_helpful_count)'))
        ];
    }

    /**
     * Get location rankings
     */
    private function getLocationRankings()
    {
        return RatingStatistics::with('location')
            ->where('total_ratings', '>=', 3)
            ->orderBy('average_rating', 'desc')
            ->orderBy('total_ratings', 'desc')
            ->limit(20)
            ->get();
    }

    /**
     * Bulk actions for ratings
     */
    public function bulkAction(Request $request)
    {
        try {
            // Debug: لاگ کردن درخواست
            \Log::info('Bulk action request received:', [
                'method' => $request->method(),
                'all_data' => $request->all(),
                'content_type' => $request->header('Content-Type')
            ]);

            $validatedData = $request->validate([
                'action' => 'required|in:approve,reject,delete',
                'rating_ids' => 'required|array',
                'rating_ids.*' => 'integer'
            ]);

            \Log::info('Validation passed:', $validatedData);

            $count = 0;
            $errors = [];

            foreach ($request->rating_ids as $id) {
                try {
                    $rating = Rating::find($id);

                    if (!$rating) {
                        $errors[] = "امتیاز با ID {$id} یافت نشد";
                        continue;
                    }

                    switch ($request->action) {
                        case 'approve':
                            // Use raw SQL to avoid collation issues
                            DB::statement("UPDATE ratings SET status = 'approved', updated_at = NOW() WHERE id = ?", [$id]);
                            $count++;
                            \Log::info("Successfully approved rating {$id} using raw SQL");
                            continue 2; // Skip Eloquent save
                        case 'reject':
                            // Use raw SQL to avoid collation issues
                            DB::statement("UPDATE ratings SET status = 'rejected', updated_at = NOW() WHERE id = ?", [$id]);
                            $count++;
                            \Log::info("Successfully rejected rating {$id} using raw SQL");
                            continue 2; // Skip Eloquent save
                        case 'delete':
                            DB::statement("DELETE FROM ratings WHERE id = ?", [$id]);
                            $count++;
                            \Log::info("Successfully deleted rating {$id} using raw SQL");
                            continue 2; // Skip Eloquent save
                    }

                } catch (\Exception $e) {
                    $errors[] = "خطا در پردازش امتیاز {$id}: " . $e->getMessage();
                    \Log::error("Error processing rating {$id}: " . $e->getMessage());
                }
            }

            $actionText = [
                'approve' => 'تایید',
                'reject' => 'رد',
                'delete' => 'حذف'
            ];

            $message = "{$count} امتیاز با موفقیت {$actionText[$request->action]} شد.";

            if (!empty($errors)) {
                $message .= " خطاها: " . implode(', ', $errors);
            }

            return response()->json([
                'success' => true,
                'message' => $message,
                'processed_count' => $count,
                'errors' => $errors
            ]);

        } catch (\Illuminate\Validation\ValidationException $e) {
            \Log::error('Validation error in bulk action:', $e->errors());
            return response()->json([
                'success' => false,
                'message' => 'خطای اعتبارسنجی: ' . implode(', ', array_flatten($e->errors()))
            ], 422);
        } catch (\Exception $e) {
            \Log::error('Bulk action error:', [
                'message' => $e->getMessage(),
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString()
            ]);
            return response()->json([
                'success' => false,
                'message' => 'خطایی در انجام عملیات رخ داد: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Test method for bulk action
     */
    public function testBulkAction()
    {
        return response()->json([
            'success' => true,
            'message' => 'Test successful',
            'data' => [
                'ratings_count' => Rating::count(),
                'pending_count' => Rating::where('status', 'pending')->count()
            ]
        ]);
    }

}
