<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Storage;
use Carbon\Carbon;

class LogController extends Controller
{
    /**
     * نمایش لیست فایل‌های لاگ
     */
    public function index()
    {
        $logs = $this->getLogs();
        return view('admin.logs.index', compact('logs'));
    }

    /**
     * نمایش محتوای یک فایل لاگ
     */
    public function show($filename)
    {
        $path = storage_path('logs/' . $filename);
        
        if (!file_exists($path)) {
            return redirect()->route('admin.logs.index')
                ->with('error', 'فایل لاگ یافت نشد.');
        }

        $content = file_get_contents($path);
        return view('admin.logs.show', compact('filename', 'content'));
    }

    /**
     * دانلود فایل لاگ
     */
    public function download($filename)
    {
        $path = storage_path('logs/' . $filename);
        
        if (!file_exists($path)) {
            return redirect()->route('admin.logs.index')
                ->with('error', 'فایل لاگ یافت نشد.');
        }

        return response()->download($path);
    }

    /**
     * حذف فایل لاگ
     */
    public function destroy($filename)
    {
        try {
            $path = storage_path('logs/' . $filename);
            
            if (file_exists($path)) {
                unlink($path);
                return redirect()->route('admin.logs.index')
                    ->with('success', 'فایل لاگ با موفقیت حذف شد.');
            }

            return redirect()->route('admin.logs.index')
                ->with('error', 'فایل لاگ یافت نشد.');
        } catch (\Exception $e) {
            return redirect()->route('admin.logs.index')
                ->with('error', 'خطا در حذف فایل لاگ: ' . $e->getMessage());
        }
    }

    /**
     * دریافت لیست فایل‌های لاگ
     */
    private function getLogs()
    {
        $logs = [];
        $path = storage_path('logs');

        if (is_dir($path)) {
            $files = scandir($path);
            foreach ($files as $file) {
                if ($file != '.' && $file != '..' && pathinfo($file, PATHINFO_EXTENSION) == 'log') {
                    $logs[] = [
                        'filename' => $file,
                        'size' => $this->formatSize(filesize($path . '/' . $file)),
                        'created_at' => Carbon::createFromTimestamp(filemtime($path . '/' . $file))->format('Y-m-d H:i:s')
                    ];
                }
            }
        }

        return $logs;
    }

    /**
     * تبدیل حجم فایل به فرمت خوانا
     */
    private function formatSize($size)
    {
        $units = ['B', 'KB', 'MB', 'GB', 'TB'];
        $i = 0;
        while ($size >= 1024 && $i < count($units) - 1) {
            $size /= 1024;
            $i++;
        }
        return round($size, 2) . ' ' . $units[$i];
    }
} 