<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class SiteMenu extends Model
{
    use HasFactory;

    protected $table = 'site_menus';

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'title',
        'url',
        'icon',
        'target',
        'parent_id',
        'sort_order',
        'is_active',
        'show_for_guests',
        'show_for_users',
        'language_code',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array
     */
    protected $casts = [
        'is_active' => 'boolean',
        'show_for_guests' => 'boolean',
        'show_for_users' => 'boolean',
        'sort_order' => 'integer',
    ];

    /**
     * Get the parent menu.
     */
    public function parent()
    {
        return $this->belongsTo(SiteMenu::class, 'parent_id');
    }

    /**
     * Get the children menus.
     */
    public function children()
    {
        return $this->hasMany(SiteMenu::class, 'parent_id')->orderBy('sort_order');
    }

    /**
     * Get the language.
     */
    public function language()
    {
        return $this->belongsTo(Language::class, 'language_code', 'code');
    }

    /**
     * Scope a query to only include active menus.
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Scope a query to only include main menus (no parent).
     */
    public function scopeMain($query)
    {
        return $query->whereNull('parent_id');
    }

    /**
     * Scope a query to filter by language.
     */
    public function scopeForLanguage($query, $languageCode)
    {
        return $query->where('language_code', $languageCode);
    }

    /**
     * Scope a query to filter by user type.
     */
    public function scopeForUserType($query, $isGuest = true)
    {
        if ($isGuest) {
            return $query->where('show_for_guests', true);
        } else {
            return $query->where('show_for_users', true);
        }
    }

    /**
     * Get menu tree structure.
     */
    public static function getMenuTree($languageCode = 'fa', $isGuest = true)
    {
        $query = self::active()
            ->forLanguage($languageCode)
            ->forUserType($isGuest)
            ->main()
            ->with(['children' => function ($query) use ($isGuest) {
                $query->active()
                    ->forUserType($isGuest)
                    ->orderBy('sort_order');
            }])
            ->orderBy('sort_order');

        return $query->get();
    }

    /**
     * Check if menu has children.
     */
    public function hasChildren()
    {
        return $this->children()->count() > 0;
    }

    /**
     * Get full URL with domain if needed.
     */
    public function getFullUrlAttribute()
    {
        if (filter_var($this->url, FILTER_VALIDATE_URL)) {
            return $this->url;
        }
        
        return url($this->url);
    }
}
