<?php

namespace App\Http\Controllers;

use App\Models\Location;
use App\Models\Rating;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class ReviewController extends Controller
{
    /**
     * نمایش لیست نظرات برای یک مکان خاص
     *
     * @param  int  $location_id
     * @return \Illuminate\Http\Response
     */
    public function index($location_id)
    {
        if (! $location_id) {
            return redirect()->route('home')->with('error', 'شناسه مکان مشخص نشده است');
        }

        $location = Location::findOrFail($location_id);

        // دریافت نظرات تایید شده
        $reviews = Rating::where('title_id', $location->title_id)
            ->where('status', 'approved')
            ->orderBy('timestamp', 'desc')
            ->paginate(10);

        // دریافت خلاصه امتیازات
        $ratingSummary = $this->getRatingSummary($location_id);

        // دریافت انواع سفر برای فیلتر
        $tripTypes = Rating::where('title_id', $location->title_id)
            ->where('status', 'approved')
            ->whereNotNull('trip_type')
            ->distinct()
            ->pluck('trip_type');

        return view('reviews.index', [
            'location' => $location,
            'reviews' => $reviews,
            'ratingSummary' => $ratingSummary,
            'tripTypes' => $tripTypes,
        ]);
    }

    /**
     * نمایش فرم ثبت نظر جدید
     *
     * @param  int  $location_id
     * @return \Illuminate\Http\Response
     */
    public function create($location_id = null)
    {
        if (! $location_id) {
            return redirect()->route('home')->with('error', 'شناسه مکان مشخص نشده است');
        }

        $location = Location::findOrFail($location_id);

        // بررسی آیا کاربر قبلاً نظر ثبت کرده است
        if (Auth::check()) {
            $existingReview = Rating::where('title_id', $location->title_id)
                ->where('email', Auth::user()->email)
                ->first();

            if ($existingReview) {
                return redirect()->route('reviews.index', ['location_id' => $location_id])
                    ->with('info', 'شما قبلاً برای این مکان نظر ثبت کرده‌اید');
            }
        }

        return view('reviews.create', [
            'location' => $location,
        ]);
    }

    /**
     * ذخیره نظر جدید در دیتابیس
     *
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        // اعتبارسنجی
        $validatedData = $request->validate([
            'title_id' => 'required|exists:locations,title_id',
            'rating' => 'required|integer|min:1|max:5',
            'name' => 'required|string|max:100',
            'email' => 'nullable|email|max:100',
            'trip_type' => 'nullable|string|max:50',
            'service_rating' => 'nullable|integer|min:1|max:5',
            'cleanliness_rating' => 'nullable|integer|min:1|max:5',
            'location_rating' => 'nullable|integer|min:1|max:5',
            'price_rating' => 'nullable|integer|min:1|max:5',
            'positive_comment' => 'nullable|string',
            'negative_comment' => 'nullable|string',
            'comment' => 'nullable|string',
            'terms_accepted' => 'required|accepted',
        ]);

        // Get the location
        $location = Location::where('title_id', $request->title_id)->firstOrFail();

        // ساخت رکورد نظر
        $review = new Rating;
        $review->title_id = $request->title_id;
        $review->category = $location->category;
        $review->hotel_name = $location->title;
        $review->rating = $request->rating;
        $review->name = $request->name;
        $review->email = $request->email;
        $review->trip_type = $request->trip_type;
        $review->service_rating = $request->service_rating;
        $review->cleanliness_rating = $request->cleanliness_rating;
        $review->location_rating = $request->location_rating;
        $review->price_rating = $request->price_rating;
        $review->positive_comment = $request->positive_comment;
        $review->negative_comment = $request->negative_comment;
        $review->comment = $request->comment;
        $review->timestamp = now();

        // بررسی وضعیت تایید خودکار
        $autoApprove = config('reviews.auto_approve', false);
        $review->status = $autoApprove ? 'approved' : 'pending';

        $review->save();

        // Check if this is an AJAX request
        if ($request->ajax()) {
            return response()->json([
                'success' => true,
                'message' => $autoApprove
                    ? 'نظر شما با موفقیت ثبت شد. با تشکر از مشارکت شما!'
                    : 'نظر شما با موفقیت ثبت شد و پس از بررسی و تایید، نمایش داده خواهد شد.',
                'review_id' => $review->id,
            ]);
        }

        // For non-AJAX requests, return normal redirect
        if ($autoApprove) {
            return redirect()->route('reviews.index', ['location_id' => $location->id])
                ->with('success', 'نظر شما با موفقیت ثبت شد. با تشکر از مشارکت شما!');
        } else {
            return redirect()->route('reviews.index', ['location_id' => $location->id])
                ->with('success', 'نظر شما با موفقیت ثبت شد و پس از بررسی و تایید، نمایش داده خواهد شد.');
        }
    }

    /**
     * فیلتر کردن نظرات براساس پارامترهای ارسالی
     *
     * @return \Illuminate\Http\Response
     */
    public function filter(Request $request)
    {
        $locationId = $request->location_id;

        if (! $locationId) {
            return response()->json(['error' => 'شناسه مکان مشخص نشده است'], 400);
        }

        $location = Location::findOrFail($locationId);

        // ساخت کوئری
        $query = Rating::where('title_id', $location->title_id)
            ->where('status', 'approved')
            ->orderBy('timestamp', 'desc');

        // اعمال فیلترها
        if ($request->has('rating') && $request->rating > 0) {
            $query->where('rating', $request->rating);
        }

        if ($request->has('trip_type') && ! empty($request->trip_type)) {
            $query->where('trip_type', $request->trip_type);
        }

        if ($request->has('sort')) {
            switch ($request->sort) {
                case 'newest':
                    $query->orderBy('timestamp', 'desc');
                    break;
                case 'oldest':
                    $query->orderBy('timestamp', 'asc');
                    break;
                case 'highest':
                    $query->orderBy('rating', 'desc');
                    break;
                case 'lowest':
                    $query->orderBy('rating', 'asc');
                    break;
            }
        }

        $reviews = $query->paginate(10);

        return response()->json([
            'reviews' => $reviews,
            'links' => (string) $reviews->links(),
        ]);
    }

    /**
     * علامت‌گذاری یک نظر به‌عنوان مفید
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function markHelpful(Request $request, $id)
    {
        $review = Rating::findOrFail($id);

        // بررسی اینکه کاربر قبلاً این نظر را مفید علامت نزده باشد
        $cookie = $request->cookie('helpful_reviews', '');
        $helpfulReviews = explode(',', $cookie);

        if (in_array($id, $helpfulReviews)) {
            return response()->json(['error' => 'شما قبلاً این نظر را مفید علامت زده‌اید'], 400);
        }

        $review->save();

        // ذخیره کوکی جدید
        $helpfulReviews[] = $id;
        $newCookie = implode(',', $helpfulReviews);

        return response()->json([
            'success' => true,
        ])->cookie('helpful_reviews', $newCookie, 10080); // یک هفته
    }

    /**
     * دریافت خلاصه امتیازات برای یک مکان
     *
     * @param  int  $locationId
     * @return array
     */
    private function getRatingSummary($locationId)
    {
        $location = Location::findOrFail($locationId);

        $ratings = Rating::where('title_id', $location->title_id)
            ->where('status', 'approved')
            ->get();

        $ratingStats = [
            '5' => 0,
            '4' => 0,
            '3' => 0,
            '2' => 0,
            '1' => 0,
        ];

        $totalCount = $ratings->count();
        $totalSum = 0;
        $avgRating = 0;

        foreach ($ratings as $rating) {
            $ratingValue = (int) $rating->rating;
            if (isset($ratingStats[$ratingValue])) {
                $ratingStats[$ratingValue]++;
            }
            $totalSum += $ratingValue;
        }

        if ($totalCount > 0) {
            $avgRating = $totalSum / $totalCount;
        }

        return [
            'avg' => round($avgRating, 1),
            'total' => $totalCount,
            'stats' => $ratingStats,
            'service_avg' => $ratings->where('service_rating', '>', 0)->avg('service_rating') ?? 0,
            'cleanliness_avg' => $ratings->where('cleanliness_rating', '>', 0)->avg('cleanliness_rating') ?? 0,
            'location_avg' => $ratings->where('location_rating', '>', 0)->avg('location_rating') ?? 0,
            'price_avg' => $ratings->where('price_rating', '>', 0)->avg('price_rating') ?? 0,
        ];
    }
}
