<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\RateLimiter;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use Illuminate\Validation\ValidationException;

class ForgotPasswordController extends Controller
{
    /**
     * Show the form for requesting a password reset link.
     */
    public function showLinkRequestForm()
    {
        try {
            // Get real-time statistics for the page
            $stats = [
                'total_users' => User::count(),
                'active_users' => User::where('status', 'active')->count(),
                'total_locations' => \App\Models\Location::count(),
                'total_ratings' => \App\Models\Rating::count(),
                'online_users' => rand(15, 45), // Simulated online users
                'platform_growth' => rand(12, 28),
                'popular_locations' => [],
                'recent_activities' => []
            ];

            return view('user.auth.forgot-password', compact('stats'));
        } catch (\Exception $e) {
            \Log::error('Error loading forgot password page: ' . $e->getMessage());
            
            // Fallback stats
            $stats = [
                'total_users' => 0,
                'active_users' => 0,
                'total_locations' => 0,
                'total_ratings' => 0,
                'online_users' => 0,
                'platform_growth' => 0,
                'popular_locations' => [],
                'recent_activities' => []
            ];

            return view('user.auth.forgot-password', compact('stats'));
        }
    }

    /**
     * Send a reset link to the given user.
     */
    public function sendResetLinkEmail(Request $request)
    {
        // Rate limiting for password reset attempts
        $key = 'password_reset_' . $request->ip();
        if (RateLimiter::tooManyAttempts($key, 3)) {
            return back()->withErrors([
                'email' => 'تعداد تلاش‌های بازیابی رمز عبور بیش از حد مجاز است. لطفاً 60 دقیقه دیگر تلاش کنید.'
            ]);
        }

        $validator = Validator::make($request->all(), [
            'email' => ['required', 'email:rfc,dns', 'max:100'],
        ], [
            'email.required' => 'ایمیل الزامی است.',
            'email.email' => 'فرمت ایمیل صحیح نیست.',
            'email.max' => 'ایمیل نباید بیش از 100 کاراکتر باشد.',
        ]);

        if ($validator->fails()) {
            RateLimiter::hit($key, 3600); // 60 minutes lockout
            return back()->withErrors($validator)->withInput();
        }

        // Check if user exists and is active
        $user = User::where('email', $request->email)
                   ->where('status', 'active')
                   ->first();

        if (!$user) {
            RateLimiter::hit($key, 3600); // 60 minutes lockout
            
            // Don't reveal if email exists or not for security
            return back()->with('status', 'اگر ایمیل شما در سیستم موجود باشد، لینک بازیابی رمز عبور برای شما ارسال خواهد شد.');
        }

        // Generate secure reset token
        $token = Str::random(64);
        $expiry = now()->addHours(1); // Token expires in 1 hour

        // Store reset token in database
        \DB::table('password_reset_tokens')->updateOrInsert(
            ['email' => $request->email],
            [
                'token' => Hash::make($token),
                'created_at' => now(),
                'expires_at' => $expiry,
                'ip_address' => $request->ip(),
                'user_agent' => $request->userAgent()
            ]
        );

        // Send reset email
        try {
            $resetUrl = route('user.password.reset', ['token' => $token, 'email' => $request->email]);
            
            Mail::send('user.auth.emails.password-reset', [
                'user' => $user,
                'resetUrl' => $resetUrl,
                'expiry' => $expiry,
                'ipAddress' => $request->ip(),
                'userAgent' => $request->userAgent()
            ], function ($message) use ($user) {
                $message->to($user->email, $user->full_name)
                        ->subject('بازیابی رمز عبور - مکان‌های ایران');
            });

            // Log password reset request
            \Log::info('Password reset requested', [
                'email' => $request->email,
                'ip' => $request->ip(),
                'user_agent' => $request->userAgent(),
                'timestamp' => now()->toISOString()
            ]);

            RateLimiter::clear($key);
            
            return back()->with('status', 'لینک بازیابی رمز عبور به ایمیل شما ارسال شد. لطفاً ایمیل خود را بررسی کنید.');
            
        } catch (\Exception $e) {
            \Log::error('Failed to send password reset email', [
                'email' => $request->email,
                'error' => $e->getMessage(),
                'ip' => $request->ip()
            ]);

            return back()->withErrors([
                'email' => 'خطا در ارسال ایمیل. لطفاً دوباره تلاش کنید.'
            ]);
        }
    }

    /**
     * Show the form for resetting the password.
     */
    public function showResetForm(Request $request, $token = null)
    {
        if (!$token || !$request->email) {
            return redirect()->route('user.password.request')
                           ->withErrors(['email' => 'لینک بازیابی نامعتبر است.']);
        }

        // Check if token exists and is valid
        $resetRecord = \DB::table('password_reset_tokens')
                         ->where('email', $request->email)
                         ->first();

        if (!$resetRecord || !Hash::check($token, $resetRecord->token)) {
            return redirect()->route('user.password.request')
                           ->withErrors(['email' => 'لینک بازیابی نامعتبر یا منقضی شده است.']);
        }

        // Check if token is expired
        if (now()->gt($resetRecord->expires_at)) {
            \DB::table('password_reset_tokens')->where('email', $request->email)->delete();
            return redirect()->route('user.password.request')
                           ->withErrors(['email' => 'لینک بازیابی منقضی شده است. لطفاً درخواست جدیدی ارسال کنید.']);
        }

        return view('user.auth.reset-password', [
            'token' => $token,
            'email' => $request->email
        ]);
    }

    /**
     * Reset the given user's password.
     */
    public function reset(Request $request)
    {
        // Rate limiting for password reset attempts
        $key = 'password_reset_submit_' . $request->ip();
        if (RateLimiter::tooManyAttempts($key, 5)) {
            return back()->withErrors([
                'email' => 'تعداد تلاش‌های تغییر رمز عبور بیش از حد مجاز است. لطفاً 30 دقیقه دیگر تلاش کنید.'
            ]);
        }

        $validator = Validator::make($request->all(), [
            'token' => ['required'],
            'email' => ['required', 'email:rfc,dns'],
            'password' => [
                'required',
                'string',
                'min:8',
                'max:128',
                'confirmed',
                'regex:/^(?=.*[a-z])(?=.*[A-Z])(?=.*\d)(?=.*[@$!%*?&])[A-Za-z\d@$!%*?&]{8,}$/'
            ],
        ], [
            'token.required' => 'توکن بازیابی الزامی است.',
            'email.required' => 'ایمیل الزامی است.',
            'email.email' => 'فرمت ایمیل صحیح نیست.',
            'password.required' => 'رمز عبور الزامی است.',
            'password.min' => 'رمز عبور باید حداقل 8 کاراکتر باشد.',
            'password.regex' => 'رمز عبور باید شامل حروف بزرگ، کوچک، عدد و کاراکتر خاص باشد.',
            'password.confirmed' => 'تکرار رمز عبور با رمز عبور مطابقت ندارد.',
        ]);

        if ($validator->fails()) {
            RateLimiter::hit($key, 1800); // 30 minutes lockout
            return back()->withErrors($validator)->withInput($request->only('email'));
        }

        // Verify reset token
        $resetRecord = \DB::table('password_reset_tokens')
                         ->where('email', $request->email)
                         ->first();

        if (!$resetRecord || !Hash::check($request->token, $resetRecord->token)) {
            RateLimiter::hit($key, 1800);
            return back()->withErrors(['email' => 'لینک بازیابی نامعتبر است.']);
        }

        // Check if token is expired
        if (now()->gt($resetRecord->expires_at)) {
            \DB::table('password_reset_tokens')->where('email', $request->email)->delete();
            return back()->withErrors(['email' => 'لینک بازیابی منقضی شده است.']);
        }

        // Find user
        $user = User::where('email', $request->email)->first();
        if (!$user) {
            RateLimiter::hit($key, 1800);
            return back()->withErrors(['email' => 'کاربر یافت نشد.']);
        }

        // Check if new password is different from current password
        if (Hash::check($request->password, $user->password)) {
            return back()->withErrors(['password' => 'رمز عبور جدید نباید با رمز عبور فعلی یکسان باشد.']);
        }

        // Update password
        $user->update([
            'password' => Hash::make($request->password),
            'password_updated_at' => now(),
            'remember_token' => null, // Invalidate remember tokens
        ]);

        // Delete reset token
        \DB::table('password_reset_tokens')->where('email', $request->email)->delete();

        // Log password reset
        \Log::info('Password reset completed', [
            'user_id' => $user->id,
            'email' => $request->email,
            'ip' => $request->ip(),
            'user_agent' => $request->userAgent(),
            'timestamp' => now()->toISOString()
        ]);

        RateLimiter::clear($key);

        return redirect()->route('user.login')
                        ->with('status', 'رمز عبور شما با موفقیت تغییر یافت. اکنون می‌توانید وارد شوید.');
    }
}
