<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Message;
use Illuminate\Http\Request;

class MessageController extends Controller
{
    /**
     * نمایش لیست پیام‌ها
     */
    public function index()
    {
        $messages = Message::latest()->paginate(20);
        return view('admin.messages.index', compact('messages'));
    }

    /**
     * نمایش جزئیات یک پیام
     */
    public function show(Message $message)
    {
        // اگر پیام خوانده نشده است، آن را به عنوان خوانده شده علامت‌گذاری کن
        if (!$message->read_at) {
            $message->update(['read_at' => now()]);
        }
        return view('admin.messages.show', compact('message'));
    }

    /**
     * حذف پیام
     */
    public function destroy(Message $message)
    {
        $message->delete();
        return redirect()->back()
            ->with('success', 'پیام با موفقیت به سطل زباله منتقل شد.');
    }

    /**
     * نمایش صندوق ورودی پیام‌ها
     */
    public function inbox()
    {
        $messages = Message::where('recipient_id', auth()->id())
            ->whereNull('deleted_at')
            ->latest()
            ->paginate(20);
        return view('admin.messages.inbox', compact('messages'));
    }

    /**
     * نمایش پیام‌های ارسال شده
     */
    public function sent()
    {
        $messages = Message::where('sender_id', auth()->id())
            ->whereNull('deleted_at')
            ->latest()
            ->paginate(20);
        return view('admin.messages.sent', compact('messages'));
    }

    /**
     * نمایش پیام‌های حذف شده (سطل زباله)
     */
    public function trash()
    {
        $messages = Message::onlyTrashed()
            ->where(function($query) {
                $query->where('sender_id', auth()->id())
                    ->orWhere('recipient_id', auth()->id());
            })
            ->latest('deleted_at')
            ->paginate(20);
        return view('admin.messages.trash', compact('messages'));
    }

    /**
     * بازیابی پیام حذف شده
     */
    public function restore($id)
    {
        $message = Message::onlyTrashed()->findOrFail($id);
        
        // بررسی دسترسی کاربر به پیام
        if ($message->sender_id !== auth()->id() && $message->recipient_id !== auth()->id()) {
            abort(403);
        }
        
        $message->restore();
        return redirect()->route('admin.messages.trash')
            ->with('success', 'پیام با موفقیت بازیابی شد.');
    }

    /**
     * حذف دائمی پیام
     */
    public function forceDelete($id)
    {
        $message = Message::onlyTrashed()->findOrFail($id);
        
        // بررسی دسترسی کاربر به پیام
        if ($message->sender_id !== auth()->id() && $message->recipient_id !== auth()->id()) {
            abort(403);
        }
        
        $message->forceDelete();
        return redirect()->route('admin.messages.trash')
            ->with('success', 'پیام با موفقیت برای همیشه حذف شد.');
    }
} 