<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\DisplaySetting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;

class DisplaySettingController extends Controller
{
    /**
     * نمایش لیست تنظیمات
     */
    public function index()
    {
        $settings = DisplaySetting::orderBy('page_key')->get();

        return view('admin.display-settings.index', compact('settings'));
    }

    /**
     * نمایش فرم ایجاد تنظیمات جدید
     */
    public function create()
    {
        return view('admin.display-settings.create');
    }

    /**
     * ذخیره تنظیمات جدید
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'page_key' => 'required|string|max:50',
            'section_key' => 'required|string|max:50',
            'desktop_items' => 'required|integer|min:1',
            'mobile_items' => 'required|integer|min:1',
            'load_more_text' => 'nullable|string|max:100',
            'is_active' => 'boolean',
        ]);

        DisplaySetting::create($validated);

        // پاک کردن کش
        Cache::forget('display_settings');

        return redirect()->route('admin.display-settings.index')
            ->with('success', 'تنظیمات با موفقیت ایجاد شد.');
    }

    /**
     * نمایش فرم ویرایش تنظیمات
     */
    public function edit(DisplaySetting $displaySetting)
    {
        return view('admin.display-settings.edit', compact('displaySetting'));
    }

    /**
     * بروزرسانی تنظیمات
     */
    public function update(Request $request, DisplaySetting $displaySetting)
    {
        $validated = $request->validate([
            'page_key' => 'required|string|max:50',
            'section_key' => 'required|string|max:50',
            'desktop_items' => 'required|integer|min:1',
            'mobile_items' => 'required|integer|min:1',
            'load_more_text' => 'nullable|string|max:100',
            'is_active' => 'boolean',
        ]);

        $displaySetting->update($validated);

        // پاک کردن کش
        Cache::forget('display_settings');

        return redirect()->route('admin.display-settings.index')
            ->with('success', 'تنظیمات با موفقیت بروزرسانی شد.');
    }

    /**
     * حذف تنظیمات
     */
    public function destroy(DisplaySetting $displaySetting)
    {
        $displaySetting->delete();

        // پاک کردن کش
        Cache::forget('display_settings');

        return redirect()->route('admin.display-settings.index')
            ->with('success', 'تنظیمات با موفقیت حذف شد.');
    }

    /**
     * نمایش صفحه تنظیمات نمایش صفحه اصلی
     *
     * @return \Illuminate\View\View
     */
    public function homepage()
    {
        $settings = DisplaySetting::where('page_key', 'homepage')->first();
        return view('admin.display-settings.homepage', compact('settings'));
    }

    /**
     * نمایش تنظیمات صفحه اصلی
     *
     * @return \Illuminate\View\View
     */
    public function show()
    {
        $settings = \App\Models\DisplaySetting::where('page_key', 'homepage')->first();
        return view('admin.display-settings.homepage', compact('settings'));
    }

    /**
     * به‌روزرسانی تنظیمات صفحه اصلی
     *
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function updateHomepage(Request $request)
    {
        $validated = $request->validate([
            'hero_section' => 'required|boolean',
            'featured_section' => 'required|boolean',
            'categories_section' => 'required|boolean',
            'latest_section' => 'required|boolean',
            'popular_section' => 'required|boolean',
            'map_section' => 'required|boolean',
            'sidebar_position' => 'required|in:right,left',
            'items_per_page' => 'required|integer|min:1|max:50',
            'show_ratings' => 'required|boolean',
            'show_comments' => 'required|boolean',
            'show_share_buttons' => 'required|boolean',
        ]);

        $settings = \App\Models\DisplaySetting::updateOrCreate(
            ['page_key' => 'homepage'],
            $validated
        );

        return redirect()->route('admin.display-settings.homepage')
            ->with('success', 'تنظیمات صفحه اصلی با موفقیت به‌روزرسانی شد.');
    }
}
