<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\CategoryTemplate;
use App\Models\Category;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;

class CategoryTemplateController extends Controller
{
    /**
     * نمایش لیست قالب‌های دسته‌بندی
     */
    public function index()
    {
        $templates = CategoryTemplate::with('category')->get();
        return view('admin.category_templates.index', compact('templates'));
    }

    /**
     * نمایش فرم ایجاد قالب جدید
     */
    public function create()
    {
        $categories = Category::all();
        $templates = CategoryTemplate::all();
        return view('admin.category_templates.create', compact('categories', 'templates'));
    }

    /**
     * ذخیره قالب جدید
     */
    public function store(Request $request)
    {
        $rules = [
            'category_id' => 'required|exists:categories,id',
            'template_name' => 'required|string|max:255',
            // 'template_key' => 'required|string|max:255|unique:category_templates,template_key', // حذف شد، به صورت خودکار تولید می‌شود
            'template_type' => 'required|string|in:' . implode(',', array_keys(config('templates.types', []))),
            'parent_template_id' => 'nullable|exists:category_templates,id',
            'header_style' => 'required|string|in:' . implode(',', array_keys(config('templates.header_styles', []))),
            'content_layout' => 'required|string|in:' . implode(',', array_keys(config('templates.content_layouts', []))),
            'primary_color' => ['required', 'string', 'regex:/^#([a-fA-F0-9]{6}|[a-fA-F0-9]{3})$/'],
            'secondary_color' => ['required', 'string', 'regex:/^#([a-fA-F0-9]{6}|[a-fA-F0-9]{3})$/'],
            'icon_set' => 'required|string|in:' . implode(',', array_keys(config('templates.icon_sets', []))),
            'background_image' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'features' => 'nullable|array',
            'features.*.is_enabled' => 'nullable|string|in:1',
            'custom_css' => 'nullable|string',
            'custom_js' => 'nullable|string',
        ];

        $messages = [
            'primary_color.regex' => 'فرمت رنگ اصلی نامعتبر است. مثال: #FF0000',
            'secondary_color.regex' => 'فرمت رنگ فرعی نامعتبر است. مثال: #00FF00',
            // Add other custom messages in Farsi if needed
        ];

        $validator = Validator::make($request->all(), $rules, $messages);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $validatedData = $validator->validated();

        // Generate template_key from template_name
        $baseKey = \Illuminate\Support\Str::slug($validatedData['template_name'], '-', null); // null defaults to 'en' for Str::ascii

        // If slug is empty (e.g., template_name was purely non-ASCII chars not transliterated by 'en' mode, or consisted of only separators)
        // provide a fallback to ensure template_key is not empty, which could violate NOT NULL DB constraints.
        if (empty($baseKey)) {
            // Using 'template' and a random suffix to create a base for uniqueness check.
            // Using Str::lower for consistency as slugs are typically lowercase.
            $baseKey = 'template-' . \Illuminate\Support\Str::lower(\Illuminate\Support\Str::random(4));
        }

        $templateKey = $baseKey;
        $counter = 1;
        // Ensure template_key is unique by appending a number if necessary.
        // This loop handles cases where $baseKey itself is not unique, or if the generated $templateKey (e.g. $baseKey . '-' . $counter) is not unique.
        while (CategoryTemplate::where('template_key', $templateKey)->exists()) {
            $templateKey = $baseKey . '-' . $counter++;
        }
        $validatedData['template_key'] = $templateKey;

        $saveData = $validatedData;

        // Process features
        $processedFeatures = [];
        if (isset($validatedData['features']) && is_array($validatedData['features'])) {
            $validFeatureKeys = array_keys(config('templates.features', []));
            foreach ($validatedData['features'] as $featureKey => $featureDetails) {
                if (in_array($featureKey, $validFeatureKeys) && isset($featureDetails['is_enabled']) && $featureDetails['is_enabled'] === '1') {
                    $processedFeatures[] = $featureKey;
                }
            }
        }
        $saveData['features'] = $processedFeatures;

        // Handle background image upload
        if (isset($validatedData['background_image']) && $validatedData['background_image'] instanceof \Illuminate\Http\UploadedFile) {
            $saveData['background_image'] = $validatedData['background_image']->store('templates/backgrounds', 'public');
        } else {
            unset($saveData['background_image']);
        }

        try {
            CategoryTemplate::create($saveData);
        } catch (\Illuminate\Database\QueryException $e) {
            // نمایش جزئیات خطا برای دیباگ
            // Log::error('Error saving category template: ' . $e->getMessage());
            // Log::error('SQL: ' . $e->getSql());
            // Log::error('Bindings: ' . implode(', ', $e->getBindings()));
            return redirect()->back()
                ->with('error', 'خطا در ذخیره سازی قالب در پایگاه داده: ' . $e->getMessage())
                ->withInput();
        } catch (\Exception $e) {
            // Log::error('General error saving category template: ' . $e->getMessage());
            return redirect()->back()
                ->with('error', 'بروز خطا در هنگام ذخیره سازی قالب: ' . $e->getMessage())
                ->withInput();
        }

        return redirect()->route('admin.category-templates.index')
            ->with('success', 'قالب با موفقیت ایجاد شد.');
    }

    /**
     * نمایش جزئیات قالب
     */
    public function show(CategoryTemplate $categoryTemplate)
    {
        $categoryTemplate->load('category');
        $features = config('templates.features');
        $headerStyles = config('templates.header_styles');
        $contentLayouts = config('templates.content_layouts');
        $iconSets = config('templates.icon_sets');

        return view('admin.category_templates.show', compact(
            'categoryTemplate',
            'features',
            'headerStyles',
            'contentLayouts',
            'iconSets'
        ));
    }

    /**
     * نمایش فرم ویرایش قالب
     */
    public function edit(CategoryTemplate $template)
    {
        $categories = Category::all();
        $templates = CategoryTemplate::where('id', '!=', $template->id)->get();
        return view('admin.category_templates.edit', compact('template', 'categories', 'templates'));
    }

    /**
     * بروزرسانی قالب
     */
    public function update(Request $request, CategoryTemplate $template)
    {
        $validator = Validator::make($request->all(), [
            'category_id' => 'required|exists:categories,id',
            'template_name' => 'required|string|max:255',
            'template_type' => 'required|string|in:' . implode(',', array_keys(config('templates.types'))),
            'parent_template_id' => 'nullable|exists:category_templates,id',
            'header_style' => 'required|string|in:' . implode(',', array_keys(config('templates.header_styles'))),
            'content_layout' => 'required|string|in:' . implode(',', array_keys(config('templates.content_layouts'))),
            'primary_color' => 'required|string|max:7',
            'secondary_color' => 'required|string|max:7',
            'icon_set' => 'required|string|in:' . implode(',', array_keys(config('templates.icon_sets'))),
            'background_image' => 'nullable|image|max:2048',
            'features' => 'array',
            'features.*' => 'string|in:' . implode(',', array_keys(config('templates.features'))),
            'custom_css' => 'nullable|string',
            'custom_js' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $data = $request->all();

        if ($request->hasFile('background_image')) {
            if ($template->background_image) {
                Storage::disk('public')->delete($template->background_image);
            }
            $data['background_image'] = $request->file('background_image')->store('templates/backgrounds', 'public');
        }

        $template->update($data);

        return redirect()->route('category-templates.index')
            ->with('success', 'قالب با موفقیت بروزرسانی شد.');
    }

    /**
     * حذف قالب
     */
    public function destroy(CategoryTemplate $template)
    {
        if ($template->background_image) {
            Storage::disk('public')->delete($template->background_image);
        }

        $template->delete();

        return response()->json([
            'success' => true,
            'message' => 'قالب با موفقیت حذف شد.'
        ]);
    }

    /**
     * تغییر وضعیت فعال/غیرفعال قالب
     */
    public function toggleActive(CategoryTemplate $template)
    {
        $template->update(['is_active' => !$template->is_active]);

        return response()->json([
            'success' => true,
            'message' => 'وضعیت قالب با موفقیت تغییر کرد.'
        ]);
    }
}
